/**
 * Test Framework
 * 
 * Framework zum Ausführen und Verwalten von SCORM-Tests
 * 
 * @version 1.0
 * @author SCORM Test System
 */

class TestFramework {
    constructor(scormAPI) {
        this.api = scormAPI;
        this.tests = [];
        this.results = [];
        this.running = false;
    }
    
    /**
     * Registriert einen neuen Test
     */
    test(name, category, fn) {
        this.tests.push({
            name: name,
            category: category,
            fn: fn,
            status: 'pending'
        });
    }
    
    /**
     * Führt alle Tests aus
     */
    async runAll() {
        if (this.running) {
            console.warn('[Test Framework] Tests laufen bereits');
            return;
        }
        
        this.running = true;
        this.results = [];
        
        console.log(`[Test Framework] Starte ${this.tests.length} Tests...`);
        
        for (const test of this.tests) {
            const result = await this.runTest(test);
            this.results.push(result);
        }
        
        this.running = false;
        
        console.log('[Test Framework] Alle Tests abgeschlossen');
        
        return this.generateReport();
    }
    
    /**
     * Führt einen einzelnen Test aus
     */
    async runTest(test) {
        const start = Date.now();
        
        try {
            // Test ausführen
            await test.fn(this.api);
            
            return {
                name: test.name,
                category: test.category,
                status: 'passed',
                duration: Date.now() - start,
                error: null
            };
        } catch (error) {
            return {
                name: test.name,
                category: test.category,
                status: 'failed',
                duration: Date.now() - start,
                error: error.message || String(error)
            };
        }
    }
    
    /**
     * Generiert einen Test-Report
     */
    generateReport() {
        const passed = this.results.filter(r => r.status === 'passed').length;
        const failed = this.results.filter(r => r.status === 'failed').length;
        const total = this.results.length;
        
        return {
            total: total,
            passed: passed,
            failed: failed,
            successRate: total > 0 ? ((passed / total) * 100).toFixed(2) + '%' : '0%',
            results: this.results,
            scormLog: this.api.getLog(),
            apiInfo: this.api.getAPIInfo(),
            timestamp: new Date().toISOString()
        };
    }
    
    /**
     * Exportiert den Report als JSON
     */
    exportReport() {
        const report = this.generateReport();
        return JSON.stringify(report, null, 2);
    }
    
    /**
     * Gibt Statistiken zurück
     */
    getStatistics() {
        const byCategory = {};
        
        this.results.forEach(result => {
            if (!byCategory[result.category]) {
                byCategory[result.category] = {
                    total: 0,
                    passed: 0,
                    failed: 0
                };
            }
            
            byCategory[result.category].total++;
            if (result.status === 'passed') {
                byCategory[result.category].passed++;
            } else {
                byCategory[result.category].failed++;
            }
        });
        
        return {
            byCategory: byCategory,
            total: this.results.length,
            passed: this.results.filter(r => r.status === 'passed').length,
            failed: this.results.filter(r => r.status === 'failed').length
        };
    }
}

// Global verfügbar machen
window.TestFramework = TestFramework;

