/**
 * SCORM Test API Wrapper
 * 
 * Wrapper für SCORM 1.2 API mit automatischer API-Erkennung und Logging
 * 
 * @version 1.0
 * @author SCORM Test System
 */

class SCORMTestAPI {
    constructor() {
        this.api = null;
        this.version = null;
        this.initialized = false;
        this.log = [];
        this.detectAPI();
    }
    
    /**
     * SCORM API automatisch erkennen
     */
    detectAPI() {
        // Standard SCORM API-Find-Algorithmus
        if (typeof window.API !== 'undefined' && window.API !== null) {
            this.api = window.API;
            this.version = '1.2';
            console.log('[SCORM Test API] SCORM 1.2 API gefunden');
            return;
        }
        
        if (typeof window.API_1484_11 !== 'undefined' && window.API_1484_11 !== null) {
            this.api = window.API_1484_11;
            this.version = '2004';
            console.log('[SCORM Test API] SCORM 2004 API gefunden');
            return;
        }
        
        // Prüfe parent-Frames
        try {
            if (window.parent && typeof window.parent.API !== 'undefined' && window.parent.API !== null) {
                this.api = window.parent.API;
                this.version = '1.2';
                console.log('[SCORM Test API] SCORM 1.2 API in parent gefunden');
                return;
            }
        } catch (e) {
            // Cross-origin-Fehler ignorieren
        }
        
        // Prüfe top-Frame
        try {
            if (window.top && typeof window.top.API !== 'undefined' && window.top.API !== null) {
                this.api = window.top.API;
                this.version = '1.2';
                console.log('[SCORM Test API] SCORM 1.2 API in top gefunden');
                return;
            }
        } catch (e) {
            // Cross-origin-Fehler ignorieren
        }
        
        console.warn('[SCORM Test API] Keine SCORM API gefunden');
    }
    
    /**
     * Prüft ob API verfügbar ist
     */
    isAvailable() {
        return this.api !== null;
    }
    
    /**
     * Prüft ob API initialisiert ist
     */
    isInitialized() {
        return this.initialized;
    }
    
    /**
     * Loggt einen API-Call
     */
    logCall(method, args, result, error = null) {
        this.log.push({
            method: method,
            args: Array.from(args || []),
            result: result,
            error: error,
            timestamp: Date.now(),
            time: new Date().toISOString()
        });
    }
    
    /**
     * LMSInitialize - Initialisiert die SCORM-Session
     */
    initialize() {
        if (!this.isAvailable()) {
            const error = 'SCORM API nicht verfügbar';
            this.logCall('LMSInitialize', [], null, error);
            return { success: false, error: error };
        }
        
        try {
            const result = this.api.LMSInitialize('');
            const success = result === 'true';
            
            if (success) {
                this.initialized = true;
            }
            
            this.logCall('LMSInitialize', [''], result, success ? null : 'Initialize fehlgeschlagen');
            
            return {
                success: success,
                result: result,
                error: success ? null : this.getLastError()
            };
        } catch (e) {
            const error = 'Exception: ' + e.message;
            this.logCall('LMSInitialize', [''], null, error);
            return { success: false, error: error };
        }
    }
    
    /**
     * LMSGetValue - Liest einen Wert aus dem CMI-Datenmodell
     */
    getValue(key) {
        if (!this.isAvailable()) {
            const error = 'SCORM API nicht verfügbar';
            this.logCall('LMSGetValue', [key], null, error);
            return { success: false, error: error, value: null };
        }
        
        if (!this.initialized) {
            const error = 'API nicht initialisiert';
            this.logCall('LMSGetValue', [key], null, error);
            return { success: false, error: error, value: null };
        }
        
        try {
            const result = this.api.LMSGetValue(key);
            const success = result !== null && result !== undefined;
            
            this.logCall('LMSGetValue', [key], result, success ? null : 'GetValue fehlgeschlagen');
            
            return {
                success: success,
                value: result,
                error: success ? null : this.getLastError()
            };
        } catch (e) {
            const error = 'Exception: ' + e.message;
            this.logCall('LMSGetValue', [key], null, error);
            return { success: false, error: error, value: null };
        }
    }
    
    /**
     * LMSSetValue - Setzt einen Wert im CMI-Datenmodell
     */
    setValue(key, value) {
        if (!this.isAvailable()) {
            const error = 'SCORM API nicht verfügbar';
            this.logCall('LMSSetValue', [key, value], null, error);
            return { success: false, error: error };
        }
        
        if (!this.initialized) {
            const error = 'API nicht initialisiert';
            this.logCall('LMSSetValue', [key, value], null, error);
            return { success: false, error: error };
        }
        
        try {
            const result = this.api.LMSSetValue(key, String(value));
            const success = result === 'true';
            
            this.logCall('LMSSetValue', [key, value], result, success ? null : 'SetValue fehlgeschlagen');
            
            return {
                success: success,
                result: result,
                error: success ? null : this.getLastError()
            };
        } catch (e) {
            const error = 'Exception: ' + e.message;
            this.logCall('LMSSetValue', [key, value], null, error);
            return { success: false, error: error };
        }
    }
    
    /**
     * LMSCommit - Speichert die Daten im LMS
     */
    commit() {
        if (!this.isAvailable()) {
            const error = 'SCORM API nicht verfügbar';
            this.logCall('LMSCommit', [], null, error);
            return { success: false, error: error };
        }
        
        if (!this.initialized) {
            const error = 'API nicht initialisiert';
            this.logCall('LMSCommit', [], null, error);
            return { success: false, error: error };
        }
        
        try {
            const result = this.api.LMSCommit('');
            const success = result === 'true';
            
            this.logCall('LMSCommit', [''], result, success ? null : 'Commit fehlgeschlagen');
            
            return {
                success: success,
                result: result,
                error: success ? null : this.getLastError()
            };
        } catch (e) {
            const error = 'Exception: ' + e.message;
            this.logCall('LMSCommit', [], null, error);
            return { success: false, error: error };
        }
    }
    
    /**
     * LMSFinish - Beendet die SCORM-Session
     */
    finish() {
        if (!this.isAvailable()) {
            const error = 'SCORM API nicht verfügbar';
            this.logCall('LMSFinish', [], null, error);
            return { success: false, error: error };
        }
        
        if (!this.initialized) {
            const error = 'API nicht initialisiert';
            this.logCall('LMSFinish', [], null, error);
            return { success: false, error: error };
        }
        
        try {
            const result = this.api.LMSFinish('');
            const success = result === 'true';
            
            if (success) {
                this.initialized = false;
            }
            
            this.logCall('LMSFinish', [''], result, success ? null : 'Finish fehlgeschlagen');
            
            return {
                success: success,
                result: result,
                error: success ? null : this.getLastError()
            };
        } catch (e) {
            const error = 'Exception: ' + e.message;
            this.logCall('LMSFinish', [], null, error);
            return { success: false, error: error };
        }
    }
    
    /**
     * LMSGetLastError - Gibt den letzten Fehlercode zurück
     */
    getLastError() {
        if (!this.isAvailable()) {
            return 'API nicht verfügbar';
        }
        
        try {
            return this.api.LMSGetLastError();
        } catch (e) {
            return 'Exception: ' + e.message;
        }
    }
    
    /**
     * LMSGetErrorString - Gibt die Fehlerbeschreibung zurück
     */
    getErrorString(errorCode) {
        if (!this.isAvailable()) {
            return 'API nicht verfügbar';
        }
        
        try {
            return this.api.LMSGetErrorString(errorCode);
        } catch (e) {
            return 'Exception: ' + e.message;
        }
    }
    
    /**
     * LMSGetDiagnostic - Gibt Diagnose-Informationen zurück
     */
    getDiagnostic(errorCode) {
        if (!this.isAvailable()) {
            return 'API nicht verfügbar';
        }
        
        try {
            return this.api.LMSGetDiagnostic(errorCode);
        } catch (e) {
            return 'Exception: ' + e.message;
        }
    }
    
    /**
     * Gibt alle geloggten API-Calls zurück
     */
    getLog() {
        return this.log;
    }
    
    /**
     * Exportiert den Log als JSON
     */
    exportLog() {
        return JSON.stringify({
            version: this.version,
            apiAvailable: this.isAvailable(),
            initialized: this.initialized,
            log: this.log,
            timestamp: new Date().toISOString()
        }, null, 2);
    }
    
    /**
     * Gibt Informationen über die erkannte API zurück
     */
    getAPIInfo() {
        return {
            available: this.isAvailable(),
            version: this.version,
            initialized: this.initialized,
            lmsName: this.detectLMSName()
        };
    }
    
    /**
     * Versucht den LMS-Namen zu erkennen
     */
    detectLMSName() {
        // Einfache Erkennung basierend auf URL oder User-Agent
        const url = window.location.href.toLowerCase();
        const userAgent = navigator.userAgent.toLowerCase();
        
        if (url.includes('ilias') || userAgent.includes('ilias')) {
            return 'ILIAS';
        }
        if (url.includes('moodle') || userAgent.includes('moodle')) {
            return 'Moodle';
        }
        if (url.includes('canvas')) {
            return 'Canvas';
        }
        if (url.includes('blackboard')) {
            return 'Blackboard';
        }
        
        return 'Unknown';
    }
}

// Global verfügbar machen
window.SCORMTestAPI = SCORMTestAPI;

