/**
 * Test Runner
 * 
 * Haupt-Script zum Ausführen der Tests
 * 
 * @version 1.0
 * @author SCORM Test System
 */

(function() {
    'use strict';
    
    // Globale Variablen
    let scormAPI = null;
    let testFramework = null;
    let testReport = null;
    
    /**
     * Initialisierung beim Laden der Seite
     */
    function init() {
        console.log('[Test Runner] Initialisiere...');
        
        // SCORM API initialisieren
        scormAPI = new SCORMTestAPI();
        
        // API-Informationen anzeigen
        updateAPIInfo();
        
        // Test Framework initialisieren
        testFramework = new TestFramework(scormAPI);
        
        // Tests registrieren
        if (typeof registerBasicTests === 'function') {
            registerBasicTests(testFramework);
            console.log('[Test Runner] Basis-Tests registriert');
        }
        
        // Event-Handler
        setupEventHandlers();
        
        console.log('[Test Runner] Initialisierung abgeschlossen');
    }
    
    /**
     * Aktualisiert die API-Informationen in der UI
     */
    function updateAPIInfo() {
        const apiInfo = scormAPI.getAPIInfo();
        
        const apiStatusEl = document.getElementById('api-status');
        const scormVersionEl = document.getElementById('scorm-version');
        const lmsNameEl = document.getElementById('lms-name');
        const apiWarningEl = document.getElementById('api-warning');
        
        // API-Status aktualisieren
        if (apiInfo.available) {
            apiStatusEl.textContent = '✅ Verfügbar';
            apiStatusEl.style.color = '#27ae60';
            if (apiWarningEl) {
                apiWarningEl.style.display = 'none';
            }
        } else {
            apiStatusEl.textContent = '❌ Nicht verfügbar';
            apiStatusEl.style.color = '#e74c3c';
            if (apiWarningEl) {
                apiWarningEl.style.display = 'block';
            }
        }
        
        scormVersionEl.textContent = apiInfo.version || 'Unbekannt';
        lmsNameEl.textContent = apiInfo.lmsName || 'Unbekannt';
    }
    
    /**
     * Event-Handler einrichten
     */
    function setupEventHandlers() {
        // Tests ausführen
        document.getElementById('run-tests').addEventListener('click', async () => {
            await runTests();
        });
        
        // Log exportieren
        document.getElementById('export-log').addEventListener('click', () => {
            exportLog();
        });
        
        // Report exportieren
        document.getElementById('export-report').addEventListener('click', () => {
            exportReport();
        });
    }
    
    /**
     * Führt alle Tests aus
     */
    async function runTests() {
        console.log('[Test Runner] Starte Tests...');
        
        // UI aktualisieren
        document.getElementById('run-tests').disabled = true;
        document.getElementById('test-results').style.display = 'none';
        document.getElementById('loading').style.display = 'block';
        
        try {
            // Tests ausführen
            testReport = await testFramework.runAll();
            
            // Ergebnisse anzeigen
            displayResults(testReport);
            
            console.log('[Test Runner] Tests abgeschlossen', testReport);
        } catch (error) {
            console.error('[Test Runner] Fehler beim Ausführen der Tests:', error);
            alert('Fehler beim Ausführen der Tests: ' + error.message);
        } finally {
            // UI zurücksetzen
            document.getElementById('run-tests').disabled = false;
            document.getElementById('loading').style.display = 'none';
        }
    }
    
    /**
     * Zeigt die Test-Ergebnisse an
     */
    function displayResults(report) {
        // Zusammenfassung
        const summary = document.getElementById('test-summary');
        summary.innerHTML = `
            <div class="summary-item">
                <div class="label">Gesamt</div>
                <div class="value">${report.total}</div>
            </div>
            <div class="summary-item">
                <div class="label">Bestanden</div>
                <div class="value passed">${report.passed}</div>
            </div>
            <div class="summary-item">
                <div class="label">Fehlgeschlagen</div>
                <div class="value failed">${report.failed}</div>
            </div>
            <div class="summary-item">
                <div class="label">Erfolgsrate</div>
                <div class="value">${report.successRate}</div>
            </div>
        `;
        
        // Details
        const details = document.getElementById('test-details');
        details.innerHTML = report.results.map(result => {
            const statusIcon = result.status === 'passed' ? '✅' : '❌';
            const statusClass = result.status === 'passed' ? 'passed' : 'failed';
            
            return `
                <div class="test-result ${statusClass}">
                    <div class="test-result-header">
                        <span class="test-result-name">${result.name}</span>
                        <span class="test-result-status">${statusIcon}</span>
                    </div>
                    <div class="test-result-category">Kategorie: ${result.category}</div>
                    <div class="test-result-duration">Dauer: ${result.duration}ms</div>
                    ${result.error ? `<div class="test-result-error">Fehler: ${result.error}</div>` : ''}
                </div>
            `;
        }).join('');
        
        // Ergebnisse anzeigen
        document.getElementById('test-results').style.display = 'block';
    }
    
    /**
     * Exportiert den Log
     */
    function exportLog() {
        if (!scormAPI) {
            alert('API nicht initialisiert');
            return;
        }
        
        const log = scormAPI.exportLog();
        downloadFile(log, 'scorm-test-log.json', 'application/json');
    }
    
    /**
     * Exportiert den Report
     */
    function exportReport() {
        if (!testReport) {
            alert('Kein Report verfügbar. Bitte Tests zuerst ausführen.');
            return;
        }
        
        const report = testFramework.exportReport();
        downloadFile(report, 'scorm-test-report.json', 'application/json');
    }
    
    /**
     * Lädt eine Datei herunter
     */
    function downloadFile(content, filename, mimeType) {
        const blob = new Blob([content], { type: mimeType });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }
    
    // Initialisierung beim Laden der Seite
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();

